<?php
/*
Plugin Name: NivaSMS for WooCommerce
Description: ارسال پیامک‌های اطلاع‌رسانی ووکامرس و وردپرس از طریق وب‌سرویس NivaSMS.
Version: 1.0.0
Author: NivaSMS
License: GPLv2 or later
Text Domain: nivasms
*/

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('NivaSMS_WooCommerce')):

final class NivaSMS_WooCommerce
{
    const OPTION_KEY = 'nivasms_settings';
    const NONCE_ACTION = 'nivasms_save_settings';

    public static function init(): void
    {
        add_action('admin_menu', [__CLASS__, 'admin_menu']);
        add_action('admin_init', [__CLASS__, 'handle_post']);

        // WooCommerce
        add_action('woocommerce_new_order', [__CLASS__, 'on_new_order'], 10, 1);
        add_action('woocommerce_order_status_changed', [__CLASS__, 'on_order_status_changed'], 10, 4);

        // WordPress
        add_action('user_register', [__CLASS__, 'on_user_register'], 10, 1);
        add_action('after_password_reset', [__CLASS__, 'on_password_reset'], 10, 2);

        add_action('admin_notices', [__CLASS__, 'admin_notices']);
    }

    public static function admin_notices(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Only show notice on our page
        $page = isset($_GET['page']) ? sanitize_text_field((string)$_GET['page']) : '';
        if ($page !== 'nivasms-settings') {
            return;
        }

        if (!class_exists('WooCommerce')) {
            echo '<div class="notice notice-warning"><p>';
            echo esc_html__('افزونه WooCommerce فعال نیست. بخش ارسال پیامک سفارش‌ها فقط در صورت فعال بودن ووکامرس کار می‌کند.', 'nivasms');
            echo '</p></div>';
        }
    }

    public static function admin_menu(): void
    {
        $parent = class_exists('WooCommerce') ? 'woocommerce' : 'options-general.php';
        add_submenu_page(
            $parent,
            __('تنظیمات NivaSMS', 'nivasms'),
            __('NivaSMS', 'nivasms'),
            'manage_options',
            'nivasms-settings',
            [__CLASS__, 'render_settings_page']
        );
    }

    private static function defaults(): array
    {
        return [
            'base_url' => 'https://nivasms.ir/v1',
            'api_key' => '',
            'from_number' => '',
            'admin_phones' => '',

            'enable_admin_new_order' => 1,
            'enable_customer_processing' => 1,
            'enable_customer_completed' => 1,
            'enable_customer_cancelled' => 0,
            'enable_customer_refunded' => 0,
            'enable_customer_failed' => 0,

            'enable_user_register' => 0,
            'enable_password_reset' => 0,

            'tpl_admin_new_order' => 'سفارش جدید #{order_id} در {site_name} ثبت شد. مبلغ: {total}',
            'tpl_processing' => 'سفارش #{order_id} در حال پردازش است. {site_name}',
            'tpl_completed' => 'سفارش #{order_id} تکمیل شد. {site_name}',
            'tpl_cancelled' => 'سفارش #{order_id} لغو شد. {site_name}',
            'tpl_refunded' => 'سفارش #{order_id} مرجوع شد. {site_name}',
            'tpl_failed' => 'سفارش #{order_id} ناموفق بود. {site_name}',

            'tpl_user_register' => 'به {site_name} خوش آمدید. حساب کاربری شما ایجاد شد.',
            'tpl_password_reset' => 'رمز عبور شما در {site_name} با موفقیت تغییر کرد.',
        ];
    }

    private static function get_settings(): array
    {
        $opt = get_option(self::OPTION_KEY, []);
        if (!is_array($opt)) {
            $opt = [];
        }
        return array_merge(self::defaults(), $opt);
    }

    private static function update_settings(array $settings): void
    {
        update_option(self::OPTION_KEY, $settings);
    }

    public static function handle_post(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (empty($_POST['nivasms_action'])) {
            return;
        }

        $action = sanitize_text_field((string)$_POST['nivasms_action']);

        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce((string)$_POST['_wpnonce'], self::NONCE_ACTION)) {
            add_settings_error('nivasms', 'nivasms_csrf', __('خطای امنیتی (Nonce).', 'nivasms'), 'error');
            return;
        }

        if ($action === 'save') {
            $settings = self::get_settings();

            $settings['base_url'] = esc_url_raw(trim((string)($_POST['base_url'] ?? '')));
            $settings['api_key'] = sanitize_text_field((string)($_POST['api_key'] ?? ''));
            $settings['from_number'] = sanitize_text_field((string)($_POST['from_number'] ?? ''));
            $settings['admin_phones'] = sanitize_text_field((string)($_POST['admin_phones'] ?? ''));

            $settings['enable_admin_new_order'] = !empty($_POST['enable_admin_new_order']) ? 1 : 0;
            $settings['enable_customer_processing'] = !empty($_POST['enable_customer_processing']) ? 1 : 0;
            $settings['enable_customer_completed'] = !empty($_POST['enable_customer_completed']) ? 1 : 0;
            $settings['enable_customer_cancelled'] = !empty($_POST['enable_customer_cancelled']) ? 1 : 0;
            $settings['enable_customer_refunded'] = !empty($_POST['enable_customer_refunded']) ? 1 : 0;
            $settings['enable_customer_failed'] = !empty($_POST['enable_customer_failed']) ? 1 : 0;

            $settings['enable_user_register'] = !empty($_POST['enable_user_register']) ? 1 : 0;
            $settings['enable_password_reset'] = !empty($_POST['enable_password_reset']) ? 1 : 0;

            $settings['tpl_admin_new_order'] = wp_kses_post((string)($_POST['tpl_admin_new_order'] ?? ''));
            $settings['tpl_processing'] = wp_kses_post((string)($_POST['tpl_processing'] ?? ''));
            $settings['tpl_completed'] = wp_kses_post((string)($_POST['tpl_completed'] ?? ''));
            $settings['tpl_cancelled'] = wp_kses_post((string)($_POST['tpl_cancelled'] ?? ''));
            $settings['tpl_refunded'] = wp_kses_post((string)($_POST['tpl_refunded'] ?? ''));
            $settings['tpl_failed'] = wp_kses_post((string)($_POST['tpl_failed'] ?? ''));
            $settings['tpl_user_register'] = wp_kses_post((string)($_POST['tpl_user_register'] ?? ''));
            $settings['tpl_password_reset'] = wp_kses_post((string)($_POST['tpl_password_reset'] ?? ''));

            self::update_settings($settings);
            add_settings_error('nivasms', 'nivasms_saved', __('تنظیمات ذخیره شد.', 'nivasms'), 'updated');
        }

        if ($action === 'test_sms') {
            $settings = self::get_settings();
            $to = sanitize_text_field((string)($_POST['test_phone'] ?? ''));
            $msg = sanitize_textarea_field((string)($_POST['test_message'] ?? ''));

            $res = self::send_sms($to, $msg, [
                'base_url' => $settings['base_url'],
                'api_key' => $settings['api_key'],
                'from_number' => $settings['from_number'],
            ]);

            if (is_wp_error($res)) {
                add_settings_error('nivasms', 'nivasms_test_fail', $res->get_error_message(), 'error');
            } else {
                add_settings_error('nivasms', 'nivasms_test_ok', __('ارسال تست انجام شد. (نتیجه را در گزارش‌های پنل بررسی کنید)', 'nivasms'), 'updated');
            }
        }
    }

    public static function render_settings_page(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $settings = self::get_settings();

        settings_errors('nivasms');

        $placeholders = '{order_id}, {status}, {total}, {first_name}, {last_name}, {phone}, {site_name}';

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('تنظیمات NivaSMS', 'nivasms') . '</h1>';

        echo '<form method="post">';
        wp_nonce_field(self::NONCE_ACTION);
        echo '<input type="hidden" name="nivasms_action" value="save">';

        echo '<h2 class="title">' . esc_html__('اتصال به وب‌سرویس', 'nivasms') . '</h2>';
        echo '<table class="form-table" role="presentation">';

        echo '<tr><th scope="row"><label for="base_url">' . esc_html__('Base URL', 'nivasms') . '</label></th><td>';
        echo '<input name="base_url" id="base_url" type="url" class="regular-text ltr" value="' . esc_attr($settings['base_url']) . '" placeholder="https://nivasms.ir/v1">';
        echo '<p class="description">' . esc_html__('آدرس پایه API (پیشنهادی: https://nivasms.ir/v1).', 'nivasms') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row"><label for="api_key">' . esc_html__('API Key', 'nivasms') . '</label></th><td>';
        echo '<input name="api_key" id="api_key" type="password" class="regular-text ltr" value="' . esc_attr($settings['api_key']) . '" autocomplete="new-password">';
        echo '<p class="description">' . esc_html__('API Key پنل پیامکی را وارد کنید.', 'nivasms') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row"><label for="from_number">' . esc_html__('From Number', 'nivasms') . '</label></th><td>';
        echo '<input name="from_number" id="from_number" type="text" class="regular-text ltr" value="' . esc_attr($settings['from_number']) . '" placeholder="3000xxxx">';
        echo '<p class="description">' . esc_html__('شماره/خط ارسال (اختیاری، اگر پنل شما نیاز داشته باشد).', 'nivasms') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row"><label for="admin_phones">' . esc_html__('Admin Phones', 'nivasms') . '</label></th><td>';
        echo '<input name="admin_phones" id="admin_phones" type="text" class="regular-text ltr" value="' . esc_attr($settings['admin_phones']) . '" placeholder="0912...,0935...">';
        echo '<p class="description">' . esc_html__('شماره(های) مدیر برای دریافت پیامک سفارش جدید (با کاما جدا کنید).', 'nivasms') . '</p>';
        echo '</td></tr>';

        echo '</table>';

        echo '<h2 class="title">' . esc_html__('ووکامرس', 'nivasms') . '</h2>';
        echo '<table class="form-table" role="presentation">';

        self::checkbox_row('enable_admin_new_order', __('ارسال پیامک به مدیر هنگام سفارش جدید', 'nivasms'), $settings['enable_admin_new_order']);

        self::checkbox_row('enable_customer_processing', __('ارسال به مشتری: وضعیت در حال پردازش', 'nivasms'), $settings['enable_customer_processing']);
        self::checkbox_row('enable_customer_completed', __('ارسال به مشتری: وضعیت تکمیل شده', 'nivasms'), $settings['enable_customer_completed']);
        self::checkbox_row('enable_customer_cancelled', __('ارسال به مشتری: وضعیت لغو شده', 'nivasms'), $settings['enable_customer_cancelled']);
        self::checkbox_row('enable_customer_refunded', __('ارسال به مشتری: وضعیت مرجوع شده', 'nivasms'), $settings['enable_customer_refunded']);
        self::checkbox_row('enable_customer_failed', __('ارسال به مشتری: وضعیت ناموفق', 'nivasms'), $settings['enable_customer_failed']);

        echo '<tr><th scope="row">' . esc_html__('متن‌ها', 'nivasms') . '</th><td>';
        echo '<p class="description">' . esc_html__('Placeholders: ', 'nivasms') . '<code class="ltr">' . esc_html($placeholders) . '</code></p>';
        echo '</td></tr>';

        self::textarea_row('tpl_admin_new_order', __('متن مدیر: سفارش جدید', 'nivasms'), $settings['tpl_admin_new_order']);
        self::textarea_row('tpl_processing', __('متن مشتری: پردازش', 'nivasms'), $settings['tpl_processing']);
        self::textarea_row('tpl_completed', __('متن مشتری: تکمیل', 'nivasms'), $settings['tpl_completed']);
        self::textarea_row('tpl_cancelled', __('متن مشتری: لغو', 'nivasms'), $settings['tpl_cancelled']);
        self::textarea_row('tpl_refunded', __('متن مشتری: مرجوع', 'nivasms'), $settings['tpl_refunded']);
        self::textarea_row('tpl_failed', __('متن مشتری: ناموفق', 'nivasms'), $settings['tpl_failed']);

        echo '</table>';

        echo '<h2 class="title">' . esc_html__('رویدادهای وردپرس', 'nivasms') . '</h2>';
        echo '<table class="form-table" role="presentation">';

        self::checkbox_row('enable_user_register', __('ارسال پیامک پس از ثبت‌نام کاربر', 'nivasms'), $settings['enable_user_register']);
        self::textarea_row('tpl_user_register', __('متن ثبت‌نام', 'nivasms'), $settings['tpl_user_register']);

        self::checkbox_row('enable_password_reset', __('ارسال پیامک پس از تغییر رمز عبور', 'nivasms'), $settings['enable_password_reset']);
        self::textarea_row('tpl_password_reset', __('متن تغییر رمز عبور', 'nivasms'), $settings['tpl_password_reset']);

        echo '</table>';

        submit_button(__('ذخیره تنظیمات', 'nivasms'));
        echo '</form>';

        // Test form
        echo '<hr>';
        echo '<h2 class="title">' . esc_html__('ارسال تست', 'nivasms') . '</h2>';
        echo '<form method="post">';
        wp_nonce_field(self::NONCE_ACTION);
        echo '<input type="hidden" name="nivasms_action" value="test_sms">';
        echo '<table class="form-table" role="presentation">';
        echo '<tr><th scope="row"><label for="test_phone">' . esc_html__('شماره موبایل', 'nivasms') . '</label></th><td>';
        echo '<input name="test_phone" id="test_phone" type="text" class="regular-text ltr" placeholder="0912...">';
        echo '</td></tr>';
        echo '<tr><th scope="row"><label for="test_message">' . esc_html__('متن', 'nivasms') . '</label></th><td>';
        echo '<textarea name="test_message" id="test_message" class="large-text" rows="3">' . esc_textarea('این یک پیام تست از NivaSMS است.') . '</textarea>';
        echo '</td></tr>';
        echo '</table>';
        submit_button(__('ارسال تست', 'nivasms'), 'secondary');
        echo '</form>';

        echo '</div>';
    }

    private static function checkbox_row(string $name, string $label, $value): void
    {
        echo '<tr><th scope="row">' . esc_html($label) . '</th><td>';
        echo '<label><input type="checkbox" name="' . esc_attr($name) . '" value="1" ' . checked(1, (int)$value, false) . '> ';
        echo esc_html__('فعال', 'nivasms') . '</label>';
        echo '</td></tr>';
    }

    private static function textarea_row(string $name, string $label, string $value): void
    {
        echo '<tr><th scope="row"><label for="' . esc_attr($name) . '">' . esc_html($label) . '</label></th><td>';
        echo '<textarea name="' . esc_attr($name) . '" id="' . esc_attr($name) . '" class="large-text" rows="2">' . esc_textarea($value) . '</textarea>';
        echo '</td></tr>';
    }

    // --------------------------
    // Events
    // --------------------------

    public static function on_new_order($order_id): void
    {
        if (!class_exists('WooCommerce')) {
            return;
        }
        $settings = self::get_settings();
        if ((int)$settings['enable_admin_new_order'] !== 1) {
            return;
        }

        $phones = self::split_phones((string)$settings['admin_phones']);
        if (empty($phones)) {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $data = self::order_placeholders($order);
        $msg = self::parse_template((string)$settings['tpl_admin_new_order'], $data);

        foreach ($phones as $p) {
            self::send_sms($p, $msg, $settings);
        }
    }

    public static function on_order_status_changed($order_id, $old_status, $new_status, $order): void
    {
        if (!class_exists('WooCommerce')) {
            return;
        }
        $settings = self::get_settings();

        $map = [
            'processing' => ['key' => 'enable_customer_processing', 'tpl' => 'tpl_processing'],
            'completed'  => ['key' => 'enable_customer_completed',  'tpl' => 'tpl_completed'],
            'cancelled'  => ['key' => 'enable_customer_cancelled',  'tpl' => 'tpl_cancelled'],
            'refunded'   => ['key' => 'enable_customer_refunded',   'tpl' => 'tpl_refunded'],
            'failed'     => ['key' => 'enable_customer_failed',     'tpl' => 'tpl_failed'],
        ];

        if (!isset($map[$new_status])) {
            return;
        }

        $cfg = $map[$new_status];
        if (empty($settings[$cfg['key']])) {
            return;
        }

        if (!$order) {
            $order = wc_get_order($order_id);
        }
        if (!$order) {
            return;
        }

        $phone = $order->get_billing_phone();
        if (!$phone) {
            return;
        }

        $data = self::order_placeholders($order);
        $data['old_status'] = (string)$old_status;
        $data['new_status'] = (string)$new_status;

        $tpl = (string)($settings[$cfg['tpl']] ?? '');
        $msg = self::parse_template($tpl, $data);
        self::send_sms($phone, $msg, $settings);
    }

    public static function on_user_register($user_id): void
    {
        $settings = self::get_settings();
        if ((int)$settings['enable_user_register'] !== 1) {
            return;
        }

        $user = get_userdata((int)$user_id);
        if (!$user) {
            return;
        }

        $phone = get_user_meta((int)$user_id, 'billing_phone', true);
        if (!$phone) {
            $phone = get_user_meta((int)$user_id, 'phone', true);
        }

        if (!$phone) {
            return;
        }

        $data = [
            'site_name' => get_bloginfo('name'),
            'first_name' => (string)($user->first_name ?? ''),
            'last_name' => (string)($user->last_name ?? ''),
            'phone' => (string)$phone,
        ];

        $msg = self::parse_template((string)$settings['tpl_user_register'], $data);
        self::send_sms((string)$phone, $msg, $settings);
    }

    public static function on_password_reset($user, $new_pass): void
    {
        $settings = self::get_settings();
        if ((int)$settings['enable_password_reset'] !== 1) {
            return;
        }

        if (!$user || !is_object($user)) {
            return;
        }

        $user_id = (int)($user->ID ?? 0);
        if (!$user_id) {
            return;
        }

        $phone = get_user_meta($user_id, 'billing_phone', true);
        if (!$phone) {
            $phone = get_user_meta($user_id, 'phone', true);
        }
        if (!$phone) {
            return;
        }

        $data = [
            'site_name' => get_bloginfo('name'),
            'first_name' => (string)($user->first_name ?? ''),
            'last_name' => (string)($user->last_name ?? ''),
            'phone' => (string)$phone,
        ];

        $msg = self::parse_template((string)$settings['tpl_password_reset'], $data);
        self::send_sms((string)$phone, $msg, $settings);
    }

    // --------------------------
    // Helpers
    // --------------------------

    private static function order_placeholders($order): array
    {
        $status = method_exists($order, 'get_status') ? (string)$order->get_status() : '';
        $total = method_exists($order, 'get_total') ? (string)$order->get_total() : '';

        return [
            'order_id' => (string)($order->get_id()),
            'status' => $status,
            'total' => $total,
            'first_name' => (string)($order->get_billing_first_name()),
            'last_name' => (string)($order->get_billing_last_name()),
            'phone' => (string)($order->get_billing_phone()),
            'site_name' => get_bloginfo('name'),
        ];
    }

    private static function parse_template(string $tpl, array $data): string
    {
        $repl = [];
        foreach ($data as $k => $v) {
            $repl['{' . $k . '}'] = (string)$v;
        }
        // common alias
        if (isset($data['phone'])) {
            $repl['{mobile}'] = (string)$data['phone'];
        }
        return strtr($tpl, $repl);
    }

    private static function split_phones(string $raw): array
    {
        $raw = str_replace(['؛', ';', ' '], [',', ',', ''], $raw);
        $parts = array_filter(array_map('trim', explode(',', $raw)));
        $out = [];
        foreach ($parts as $p) {
            $n = self::normalize_phone($p);
            if ($n !== '') {
                $out[] = $n;
            }
        }
        return array_values(array_unique($out));
    }

    private static function normalize_phone(string $phone): string
    {
        // Persian/Arabic digits
        $map = [
            '۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9',
            '٠'=>'0','١'=>'1','٢'=>'2','٣'=>'3','٤'=>'4','٥'=>'5','٦'=>'6','٧'=>'7','٨'=>'8','٩'=>'9',
        ];
        $phone = strtr($phone, $map);
        $phone = preg_replace('/[^0-9\+]/', '', $phone);
        $phone = (string)$phone;

        if ($phone === '') {
            return '';
        }

        // +98 / 0098 to 0
        if (self::starts_with($phone, '+98')) {
            $phone = '0' . substr($phone, 3);
        } elseif (self::starts_with($phone, '0098')) {
            $phone = '0' . substr($phone, 4);
        } elseif (self::starts_with($phone, '98') && strlen($phone) === 12) {
            $phone = '0' . substr($phone, 2);
        }

        return $phone;
    }

    private static function starts_with(string $haystack, string $needle): bool
    {
        if ($needle === '') return true;
        return substr($haystack, 0, strlen($needle)) === $needle;
    }

    /**
     * Send SMS using NivaSMS edge-compatible endpoint.
     *
     * @param string $to
     * @param string $message
     * @param array $settings (base_url, api_key, from_number)
     * @return true|WP_Error
     */
    public static function send_sms(string $to, string $message, array $settings)
    {
        $base = trim((string)($settings['base_url'] ?? ''));
        $key = trim((string)($settings['api_key'] ?? ''));
        $from = trim((string)($settings['from_number'] ?? ''));

        if ($base === '' || $key === '') {
            return new WP_Error('nivasms_missing', 'تنظیمات NivaSMS کامل نیست (Base URL / API Key).');
        }

        $to = self::normalize_phone($to);
        if ($to === '' || $message === '') {
            return new WP_Error('nivasms_invalid', 'شماره یا متن نامعتبر است.');
        }

        $endpoint = rtrim($base, '/') . '/api/send';

        $payload = [
            'sending_type' => 'webservice',
            'from_number' => $from,
            'message' => $message,
            'params' => [
                'recipients' => [$to],
            ],
        ];

        $args = [
            'timeout' => 15,
            'headers' => [
                'Authorization' => $key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => wp_json_encode($payload, JSON_UNESCAPED_UNICODE),
        ];

        $res = wp_remote_post($endpoint, $args);
        if (is_wp_error($res)) {
            return $res;
        }

        $code = (int)wp_remote_retrieve_response_code($res);
        $body = (string)wp_remote_retrieve_body($res);

        if ($code < 200 || $code >= 300) {
            return new WP_Error('nivasms_http', 'خطا در ارتباط با وب‌سرویس. HTTP ' . $code);
        }

        // If response is JSON, check meta.status if exists
        $json = json_decode($body, true);
        if (is_array($json) && isset($json['meta']['status']) && !$json['meta']['status']) {
            $msg = isset($json['meta']['message']) ? (string)$json['meta']['message'] : 'خطا در ارسال.';
            return new WP_Error('nivasms_api', $msg);
        }

        return true;
    }
}

NivaSMS_WooCommerce::init();

endif;
